// Tool functions for getting the data ready to parse (if necessary).

#include <stdio.h>
#include <math.h>

#include "UberCassette.h"
#include "Sample.h"
#include "WAV.h"


const unsigned char gTriggerValue = 0x10;
#define SAMPLE_ENTRIES 20

void BuildRawWaveList( struct RAW_Format format )
{
	unsigned long tPointer = 0, tWaveStart;
	bool tFirstWave = true;

	tWaveStart = tPointer;
	tPointer = FindNextWavelengthStart( format, tPointer );
	AddRawCycle( (float) tPointer - tWaveStart );

    printf( "Building RAW Wave list.\n" );

	while ( tPointer < format.size )
	{
		FindWavelength( format, &tPointer, tFirstWave, false );
		tFirstWave = !tFirstWave;
	}
	printf( "RAW Wave list done.\n" );
}

unsigned long FindNextWavelengthStart( struct RAW_Format format, unsigned long pointer )
{
	// Find the beginning of a waveform.
	unsigned long tPointer = pointer;

	while ( format.data[ tPointer ] < gTriggerValue && tPointer < format.size )
		tPointer++;

	if ( tPointer == format.size )
		return -1;	// No more waveforms here.

	while ( format.data[ tPointer ] > 0 && tPointer < format.size )
		tPointer++;

	if ( tPointer == format.size )
		return -1;	// Still no more waveforms.

	return tPointer;
}


void AddCurveSample( signed char *curve, unsigned char sample )
{
	int i;

	for ( i = 0 ; i < SAMPLE_ENTRIES-1 ; i++ )
		curve[ i ] = curve[ i + 1 ];
	curve[ i ] = sample;
}

signed char GetCurveAverage( signed char *curve )
{
	signed char tAverage;
	tAverage = ((int)curve[ SAMPLE_ENTRIES-5 ] + (int)curve[ SAMPLE_ENTRIES-4 ] + (int)curve[ SAMPLE_ENTRIES-3 ] + (int)curve[ SAMPLE_ENTRIES-2 ] + (int)curve[ SAMPLE_ENTRIES-1 ]) / 5;
	return tAverage;
}

signed char GetCurvePeak( signed char *curve )
{
	signed char tPeak=0;
	int i;

	for ( i=0 ; i<SAMPLE_ENTRIES ; i++ )
	{
		if ( tPeak < curve[ i ] )
			tPeak = curve[ i ];
	}
	return tPeak;
}

signed char GetCurveTrough( signed char *curve )
{
	signed char tTrough=0;
	int i;

	for ( i=0 ; i<SAMPLE_ENTRIES ; i++ )
	{
		if ( tTrough > curve[ i ] )
			tTrough = curve[ i ];
	}
	return tTrough;
}

int HasStartedCurve( signed char *curve )
{
	signed char tTrough = GetCurveTrough( curve );

	if ( tTrough > -gTriggerValue )
		return false;

	if ( curve[ SAMPLE_ENTRIES-1 ] > curve[ SAMPLE_ENTRIES-2 ] )
		return false;	// We're going up-hill.

	return true;
}

void AddWave( int amount, float cycles )
{
	int tCounter = 0;

	const unsigned long tCycleLengths[35] = { -1,-1,-1,-1,-1,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2 };

	for ( tCounter = 0 ; tCounter < amount ; tCounter++ )
	{
		if ( cycles < 0.0f )
		{
			printf( "ERROR in amount of cycles.\n" );
		}
		if ( cycles > 1000 )
		{
			bool test=true;
		}
		AddRawCycle( cycles );
/*		if ( 1 && cycles < 35 && gSample_ValueTable[ cycles ] != -1 )
			AddRawCycle( gThresholds[ tCycleLengths[ cycles ] ] );
		else
			AddRawCycle( cycles );
*/	}
}

void BuildRawWaveList_Cleaned_old( struct RAW_Format format )
{
	unsigned long tPointer = 0;
	signed char  tCurve[ SAMPLE_ENTRIES ];
	float tCurveAverage, tLastWave = 0.0f;
	float tCurveStart = 0, tWaveLength=0.0f;
	enum CurveState { NO_CURVE, CURVE_UP, CURVE_DOWN } tState;
	int i;
	unsigned long tPeak, tTrough, tWaves, tCurveEnd = 0;
	bool tLog = false;
	bool tLoggedWave = false;

	for ( i=0 ; i < SAMPLE_ENTRIES ; i++ )
		tCurve[ i ] = 0;

	tState = NO_CURVE;

	while ( tPointer < format.size - 1 )
	{
		if ( tPointer == 894239)
		{
			tLog = true;
		}
		AddCurveSample( tCurve, format.data[ tPointer ] );
/*
		if ( GetCurvePeak( tCurve ) < gTriggerValue && GetCurveTrough( tCurve ) > -gTriggerValue ) 
		{
			if ( tState != NO_CURVE )
			{
				int tLastByte = SAMPLE_ENTRIES-1;
				for ( tLastByte = SAMPLE_ENTRIES-1 ; tLastByte > 0 ; tLastByte-- )
				{
					if ( abs( tCurve[ tLastByte ] ) < 10 )
						break;
				}
				AddWave( 1, (float)(tPointer - tLastByte - tCurveEnd) );
				tCurveEnd = tPointer - tLastByte;
			}

			tCurveStart = 0;
			tPeak = tTrough = tWaves = 0;
			tWaveLength = 0.0f;
			tState = NO_CURVE;
		}*/

		if ( abs( GetCurvePeak( tCurve ) - GetCurveTrough( tCurve )) < gTriggerValue ) 
		{
			if ( tLoggedWave == false )
			{
				// It's just noise. Forget it.
				tState = NO_CURVE;
			}
			else
			{
				if ( tState != NO_CURVE )
				{
					int tLastByte = SAMPLE_ENTRIES-1;
					for ( tLastByte = SAMPLE_ENTRIES-1 ; tLastByte > 0 ; tLastByte-- )
					{
						if ( abs( tCurve[ tLastByte ] ) < 10 )
							break;
					}
					if ( tPointer < tLastWave )
					{
						printf(" ERROR in curve.\n" );
					}
					AddWave( 1, (float)((float)tPointer - tLastByte - tCurveEnd));
					tCurveEnd = tPointer - tLastByte;
					tLastWave = (float)tPointer;
				}

				tCurveStart = 0;
				tPeak = tTrough = tWaves = 0;
				tWaveLength = 0.0f;
				tState = NO_CURVE;
			}
		}

		switch ( tState )
		{
		case NO_CURVE:
			if ( HasStartedCurve( tCurve ) )
			{
				tState = CURVE_DOWN;
				if ( tCurveEnd )
				{
//					AddWave( 1, (float)(tPointer - tLastWave) );
					//tCurveEnd = 0;
				}
				tLoggedWave = false;
				tCurveStart = (float)(tPointer - 4);
				tCurveAverage = GetCurveAverage( tCurve );
				printf( "Curve found at %f.\n", tCurveStart );
				tPeak = tTrough = tWaves = 0;
				tWaveLength = 0.0f;
			}
			break;
		case CURVE_DOWN:
			if ( format.data[ tPointer ] > format.data[ tPointer - 1 ] && format.data[ tPointer + 1 ] > format.data[ tPointer ] )
			{
				float tOldStart = tCurveStart, tCurveHalf;
				int tTroughAmplitude=0, tPeakAmplitude=0;
				float tTargetAmplitude;
				float tClosestPoint, tClosestHalfPoint;
				unsigned long tOldTrough;
				
				tOldTrough = tTrough;
				tTrough = tPointer - 1;

				if ( tTrough && tPeak )
				{
					tTroughAmplitude = format.data[ tTrough ]; tPeakAmplitude = format.data[ tPeak ];					
				}
				tState = CURVE_UP;

				// Use this if we don't have a full wave's data.
				if ( tOldTrough )
					tWaveLength = (float)(tPointer - tOldTrough);

				if ( tPeak && tTrough )
				{
					unsigned long i;
					int tX1, tX2;
					float tGradient, tTargetY;
					tTargetAmplitude = (tPeakAmplitude + tTroughAmplitude) / 2.0f;
					tClosestPoint = (float)tTrough;
					for ( i = tPeak ; i < tTrough ; i++ )
					{
						if ( fabs( (float)format.data[ i ] - tTargetAmplitude ) < fabs((float)format.data[ (int)tClosestPoint ] - tTargetAmplitude))
							tClosestPoint = (float) i;
					}
					if ( tOldTrough && gMachineData.halfWave )
					{
						tClosestHalfPoint = (float)tPeak;
						for ( i = tPeak ; i > tOldTrough ; i-- )
						{
							if ( fabs( (float)format.data[ i ] - tTargetAmplitude ) < (float)format.data[ (int)tClosestHalfPoint ]- tTargetAmplitude )
								tClosestHalfPoint = (float) i;
						}
					}
					
					// Extrapolate the actual curve trigger.
					if ( format.data[ (int)tClosestPoint ] > tTargetAmplitude )
					{
						tX1 = (int)tClosestPoint; tX2 = (int)tClosestPoint+1;
					}
					else
					{
						tX1 = (int)tClosestPoint-1; tX2 = (int)tClosestPoint;
					}

					// y = mx + c, right? So....
					tTargetY = tTargetAmplitude - (float)format.data[ (int)tX1 ];
					tGradient = (float)format.data[ tX1 ] - (float)format.data[ tX2 ];
					if (tGradient == 0.0f )
						tClosestPoint = (float)tX1 + 0.5f;
					else
						tClosestPoint = tX1 - tTargetY / tGradient;	
					tCurveStart = tClosestPoint;
					if ( tCurveStart > tPointer )
					{
						printf( "** WARNING ** Invalid curve length.\n" );
					}

					if ( gMachineData.halfWave )
					{
						// Extrapolate the actual curve trigger.
						if ( format.data[ (int)tClosestHalfPoint ] > tTargetAmplitude )
						{
							tX1 = (int)tClosestHalfPoint-1; tX2 = (int)tClosestHalfPoint;
						}
						else
						{
							tX1 = (int)tClosestHalfPoint; tX2 = (int)tClosestHalfPoint+1;
						}

						// y = mx + c, right? So....
						tGradient = (float)format.data[ tX1 ] - (float)format.data[ tX2 ];
						if (tGradient == 0.0f )
							tClosestHalfPoint = (float)tX1 + 0.5f;
						else
							tClosestHalfPoint = tX1 - tTargetY / tGradient;	
						tCurveHalf = tClosestHalfPoint;
						if ( tCurveHalf - tOldStart > 0 )
							AddWave( 1, tCurveHalf - tOldStart );
						tOldStart = tCurveHalf;
					}
				}
				if ( tOldStart )
					tWaveLength = tCurveStart - tOldStart;
				if ( tLog )
					;//printf( "Bottom of curve at %d. OldStart: %f  CurveStart: %f  Wavelength: %f.\n", tPointer, tOldStart, tCurveStart, tWaveLength );
				if ( tWaveLength )
				{
					if ( tOldStart - tLastWave > .00001f && tLastWave > 0.0f )
						AddWave( 1, tOldStart - tLastWave );
					AddWave( 1, tWaveLength );
					tLastWave = (float)tPointer;
					tLoggedWave = true;
				}
			}
			break;
		case CURVE_UP:
			if ( format.data[ tPointer ] < format.data[ tPointer - 1 ] && format.data[ tPointer + 1 ] < format.data[ tPointer ] )
			{
				tState = CURVE_DOWN;
				if ( tPeak > 0 )
				{
					tWaveLength = (float)((tPointer-1) - tPeak);
					tPeak = tPointer - 0;
				}
				else
				{
					tPeak = tPointer - 0;
					tWaveLength = (float)((tPeak - tTrough) * 2);
				}
				tWaves++;
			}
			break;
		default:
			break;
		}

		tPointer++;
	}
}


void BuildRawWaveList_Cleaned( struct RAW_Format format )
{
	unsigned long tPointer = 0;
	signed char  tCurve[ SAMPLE_ENTRIES ];
	float tCurveAverage;
	float tCurveStart = 0, tWaveLength;
	enum CurveState { NO_CURVE, CURVE_UP, CURVE_DOWN } tState;
	int i;
	unsigned long tPeak, tTrough, tWaves, tCurveEnd = 0;
	bool tLog = false, tLoggedWave = false;
	float tLastWave = 0.0f;

	for ( i=0 ; i < SAMPLE_ENTRIES ; i++ )
		tCurve[ i ] = 0;

	tState = NO_CURVE;

	while ( tPointer < format.size - 1 )
	{
		if ( tPointer == 1230387 )
		{
		//	tLog = true;
		}
		AddCurveSample( tCurve, format.data[ tPointer ] );

		if ( abs( GetCurvePeak( tCurve ) - GetCurveTrough( tCurve )) < gTriggerValue ) 
		{
			if ( tLoggedWave == false )
			{
				// It's just noise. Forget it.
				tState = NO_CURVE;
			}
			else
			{
				if ( tState != NO_CURVE )
				{
					int tLastByte = SAMPLE_ENTRIES-1;
					for ( tLastByte = SAMPLE_ENTRIES-1 ; tLastByte > 0 ; tLastByte-- )
					{
						if ( abs( tCurve[ tLastByte ] ) < 10 )
							break;
					}
					if ( tPointer < tLastWave )
					{
						printf(" ERROR in curve.\n" );
					}
//					AddWave( 1, (float)((float)tPointer - tLastWave) );
					AddWave( 1, (float)((float)tPointer - tLastByte - tCurveEnd));
					tCurveEnd = tPointer - tLastByte;
					tLastWave = (float)tPointer;
				}

				tCurveStart = 0;
				tPeak = tTrough = tWaves = 0;
				tWaveLength = 0.0f;
				tState = NO_CURVE;
			}
		}

		switch ( tState )
		{
		case NO_CURVE:
			if ( HasStartedCurve( tCurve ) )
			{
				tState = CURVE_DOWN;
				if ( tCurveEnd )
				{
//					AddWave( 1, (float)(tPointer - tLastWave) );
					//tCurveEnd = 0;
				}
				tLoggedWave = false;
				tCurveStart = (float)(tPointer - 4);
				tCurveAverage = GetCurveAverage( tCurve );
				printf( "Curve found at %f.\n", tCurveStart );
				tPeak = tTrough = tWaves = 0;
				tWaveLength = 0.0f;
			}
			break;
		case CURVE_DOWN:
			if ( format.data[ tPointer ] > format.data[ tPointer - 1 ] && format.data[ tPointer + 1 ] > format.data[ tPointer ] )
			{
				float tOldStart = tCurveStart, tCurveHalf;
				int tTroughAmplitude=0, tPeakAmplitude=0;
				float tTargetAmplitude;
				float tClosestPoint, tClosestHalfPoint;
				unsigned long tOldTrough;
				
				tOldTrough = tTrough;
				tTrough = tPointer - 1;

				if ( tTrough && tPeak )
				{
					tTroughAmplitude = format.data[ tTrough ]; tPeakAmplitude = format.data[ tPeak ];					
				}
				tState = CURVE_UP;

				// Use this if we don't have a full wave's data.
				if ( tOldTrough )
					tWaveLength = (float)(tPointer - tOldTrough);

				if ( tPeak && tTrough )
				{
					unsigned long i;
					int tX1, tX2;
					float tGradient, tTargetY;
					tTargetAmplitude = (tPeakAmplitude + tTroughAmplitude) / 2.0f;
					tClosestPoint = (float)tTrough;
					for ( i = tPeak ; i < tTrough ; i++ )
					{
						if ( fabs( (float)format.data[ i ] - tTargetAmplitude ) < fabs((float)format.data[ (int)tClosestPoint ] - tTargetAmplitude))
							tClosestPoint = (float) i;
					}
					if ( tOldTrough && gMachineData.halfWave )
					{
						tClosestHalfPoint = (float)tPeak;
						for ( i = tPeak ; i > tOldTrough ; i-- )
						{
							if ( fabs( (float)format.data[ i ] - tTargetAmplitude ) < (float)format.data[ (int)tClosestHalfPoint ]- tTargetAmplitude )
								tClosestHalfPoint = (float) i;
						}
					}
					
					// Extrapolate the actual curve trigger.
					if ( format.data[ (int)tClosestPoint ] > tTargetAmplitude )
					{
						tX1 = (int)tClosestPoint; tX2 = (int)tClosestPoint+1;
					}
					else
					{
						tX1 = (int)tClosestPoint-1; tX2 = (int)tClosestPoint;
					}

					// y = mx + c, right? So....
					tTargetY = tTargetAmplitude - (float)format.data[ (int)tX1 ];
					tGradient = (float)format.data[ tX1 ] - (float)format.data[ tX2 ];
					if (tGradient == 0.0f )
						tClosestPoint = (float)tX1 + 0.5f;
					else
						tClosestPoint = tX1 - tTargetY / tGradient;	
					tCurveStart = tClosestPoint;
					if ( tCurveStart > tPointer )
					{
						printf( "** WARNING ** Invalid curve length.\n" );
					}

					if ( gMachineData.halfWave )
					{
						// Extrapolate the actual curve trigger.
						if ( format.data[ (int)tClosestHalfPoint ] > tTargetAmplitude )
						{
							tX1 = (int)tClosestHalfPoint-1; tX2 = (int)tClosestHalfPoint;
						}
						else
						{
							tX1 = (int)tClosestHalfPoint; tX2 = (int)tClosestHalfPoint+1;
						}

						// y = mx + c, right? So....
						tGradient = (float)format.data[ tX1 ] - (float)format.data[ tX2 ];
						if (tGradient == 0.0f )
							tClosestHalfPoint = (float)tX1 + 0.5f;
						else
							tClosestHalfPoint = tX1 - tTargetY / tGradient;	
						tCurveHalf = tClosestHalfPoint;
						if ( tCurveHalf - tOldStart > 0 )
							AddWave( 1, tCurveHalf - tOldStart );
						tOldStart = tCurveHalf;
					}
				}
				if ( tOldStart )
					tWaveLength = tCurveStart - tOldStart;
				if ( tLog )
					;//printf( "Bottom of curve at %d. OldStart: %f  CurveStart: %f  Wavelength: %f.\n", tPointer, tOldStart, tCurveStart, tWaveLength );
				if ( tWaveLength )
				{
					if ( tOldStart - tLastWave > .00001f && tLastWave > 0.0f )
						AddWave( 1, tOldStart - tLastWave );
					AddWave( 1, tWaveLength );
					tLastWave = (float)tPointer;
					tLoggedWave = true;
				}
			}
			break;
		case CURVE_UP:
			if ( format.data[ tPointer ] < format.data[ tPointer - 1 ] && format.data[ tPointer + 1 ] < format.data[ tPointer ] )
			{
				tState = CURVE_DOWN;
				if ( tPeak > 0 )
				{
					tWaveLength = (float)((tPointer-1) - tPeak);
					tPeak = tPointer - 0;
				}
				else
				{
					tPeak = tPointer - 0;
					tWaveLength = (float)((tPeak - tTrough) * 2);
				}
				tWaves++;
			}
			break;
		default:
			break;
		}

		tPointer++;
	}
}

void BuildThresholds( struct RAW_Format format, int thresholds )
{
	// Find the most common wavelength.
	unsigned char tMostCommon[ 3 ]={-1,-1,-1}, tNextPeakUp = 0, tNextPeakDown = 0;
	int i, tValue, tThresholds[ 2 ];
	bool tHasGoneLow = false;

	for ( i = 2 ; i < 127 ; i++ )
	{
		if ( gWaveLengthTable[ i ] < 0x200 )
			tHasGoneLow = true;
		if ( gWaveLengthTable[ i ] >= gWaveLengthTable[ i+1 ] && tHasGoneLow )
		{
			if ( gWaveLengthTable[ i ] > gWaveLengthTable[ tMostCommon[ 0 ] ] )
			{
				if ( thresholds == 3 )
					tMostCommon[ 2 ] = tMostCommon[ 1 ];
				tMostCommon[ 1 ] = tMostCommon[ 0 ];
				tMostCommon[ 0 ] = i;
			}
			else if ( gWaveLengthTable[ i ] > gWaveLengthTable[ tMostCommon[ 1 ] ])
			{
				if ( thresholds == 3 )
					tMostCommon[ 2 ] = tMostCommon[ 1 ];
				tMostCommon[ 1 ] = i;
			}
			else if ( gWaveLengthTable[ i ] > gWaveLengthTable[ tMostCommon[ 2 ] ])
			{
				if ( thresholds == 3 )
					tMostCommon[ 2 ] = i;
			}
			tHasGoneLow = false;
		}
	}

	tValue = 0;
	for ( i=0 ; i<3 ; i++ )
		gThresholds[ i ] = 127;

	for ( i = 0 ; i < 128 ; i++ )
	{
		if ( i == tMostCommon[ 0 ] || i == tMostCommon[ 1 ] || i == tMostCommon[ 2 ] )
		{
			gThresholds[ tValue++ ] = i;
		}
	}

	tThresholds[ 0 ] = (int)(gThresholds[ 0 ] + ( (float)gThresholds[ 1 ] - (float)gThresholds[ 0 ]) * gThresholdDivisor[ 0 ]);
	tThresholds[ 1 ] = (int)(gThresholds[ 1 ] + ( (float)gThresholds[ 2 ] - (float)gThresholds[ 1 ]) * gThresholdDivisor[ 1 ]);

	for ( i = 0 ; i < 128 ; i ++ )
		gSample_ValueTable[ i ] = -1;
	for ( i = gThresholds[ 0 ] / 2 ; i < tThresholds[ 0 ] ; i++ )	// was 3/4, and 1/4 below.
		gSample_ValueTable[ i ] = 0;
	for ( i = tThresholds[ 0 ] ; i < tThresholds[ 1 ] ; i++ )
		gSample_ValueTable[ i ] = 1;

	if ( thresholds > 2 )
	{
		for ( i = gThresholds[ 2 ] - ( gThresholds[ 2 ] - gThresholds[ 1 ] ) / 4 ; i < gThresholds[ 2 ] + ( 127 - gThresholds[ 2 ] ) / 2 ; i++ )
			gSample_ValueTable[ i ] = 2;
	}
	else
		gSample_ValueTable[ 2 ] = -1;
}

void BuildWavelengthTable( struct RAW_Format format )
{
	unsigned long tPointer = 0;
	unsigned char tWavelength = 0;

	while ( tPointer >= 0 && tPointer < format.size )
	{
		tPointer = FindNextWavelengthStart( format, tPointer );

		// Now we're pointing to the beginning of a potential wave.
		tWavelength = 128;
		while ( tWavelength > 0 )
		{
			tWavelength = FindWavelength( format, &tPointer, false, true );

			if ( tWavelength > 0 )
			{
				// We have a valid wavelength!
				gWaveLengthTable[ tWavelength ]++;
			}
		}
	}

#if 0 && defined(_DEBUG)
	// Print out the wavelengths.
	{
		int i;
		for ( i=0 ; i<128 ; i++ )
			printf( "%d: %d ", i, gWaveLengthTable[ i ] );
		printf( "\n" );
	}
#endif
}

unsigned char FindWavelength( struct RAW_Format format, unsigned long *pointer, bool firstHalfWave, bool preParse )
{
	// Find out how long before we're going under again.
	unsigned long tPointer = *pointer;
	bool tHasBeenValid = false;
	unsigned char tWavelength = 0, tUpWaveLength = 0;
	bool tWarn = false;
	int tAmplitude = 0;
	unsigned long tStartSearch = *pointer;
	bool tCurveIsNegative = false;

	while( tPointer < format.size && tHasBeenValid == false )
	{
		if ( gMachineData.halfWave )
		{
			bool tCorrectWave = false;
			// Half-waves mean use amplitude on the -ve part too, so we use 0 as a reference point.
			tAmplitude = 0;
			while ( tCorrectWave == false && tPointer < format.size )
			{
				while ( tPointer < format.size && format.data[ tPointer ] == 0 )
					tPointer++;
				if ( (firstHalfWave && format.data[ tPointer ] < 0) || (!firstHalfWave && format.data[ tPointer ] > 0) )
					tCorrectWave = true;
				else
					tPointer++;
			}
			tCurveIsNegative = firstHalfWave;
		}
		else
		{
			if ( format.data[ tPointer ] < tAmplitude )
				tAmplitude = format.data[ tPointer ];
		}
		tPointer++;

		if ( tPointer >= format.size )
		{
			*pointer = tPointer;
			return 0;
		}

		while ( tPointer < format.size && format.data[ tPointer ] == 0 )
			tPointer++;

		if ( tCurveIsNegative )
		{
			while ( tPointer < format.size && format.data[ tPointer ] < 0 )
			{
				if ( (abs(format.data[ tPointer ]) - tAmplitude ) >= gTriggerValue / 2 )
					tHasBeenValid = true;
				tPointer++;
			}
		}
		else
		{
			while ( tPointer < format.size && format.data[ tPointer ] > 0 )
			{
				if ( (format.data[ tPointer ] - tAmplitude ) >= gTriggerValue / (gMachineData.halfWave ? 2 : 1) )
					tHasBeenValid = true;
				tPointer++;
			}
		}

		if ( tPointer == format.size )
		{
			*pointer = tPointer;
			return 0;
		}
	}

	tWavelength = (unsigned char)(tPointer - *pointer);

	// It's valid! Yay!
	*pointer = tPointer;

	if ( preParse == false )
	{
		if ( tPointer - tWavelength != tStartSearch )
			AddWave( 1, (float)(tPointer - tWavelength - tStartSearch) );
		AddWave( 1, tWavelength );
	}
	return tWavelength;
}

unsigned long ConvertToCycles( float samples )
{
	float tLengthInSeconds;
	unsigned long tCycles;
	unsigned long tSpeed;

	tLengthInSeconds = samples / gWAVFormat.sampleRate;
	if ( gVideoType == PAL )
		tSpeed = gMachineData.speed_PAL;
	else
		tSpeed = gMachineData.speed_NTSC;
	tCycles = (unsigned long)(((float)tSpeed) * tLengthInSeconds);
	
	return tCycles;
}
